document.addEventListener('DOMContentLoaded', function() {
  console.log('Educate Maa website loaded');
  
  document.getElementById('currentYear').textContent = new Date().getFullYear();
  
  initNavigation();
  initForms();
  
  if (document.querySelector('.stat-number')) {
    initCounters();
  }
  
  initSmoothScroll();
});

function initNavigation() {
  const currentPage = window.location.pathname.split('/').pop() || 'index.html';
  const navLinks = document.querySelectorAll('.main-nav a');
  
  navLinks.forEach(link => {
    const linkPage = link.getAttribute('href');
    if (linkPage === currentPage || (currentPage === '' && linkPage === 'index.html')) {
      link.classList.add('active');
    }
  });
  
  const mobileMenuBtn = document.querySelector('.mobile-menu-btn');
  const mainNav = document.querySelector('.main-nav');
  
  if (mobileMenuBtn && mainNav) {
    mobileMenuBtn.addEventListener('click', () => {
      const isVisible = mainNav.style.display === 'flex';
      mainNav.style.display = isVisible ? 'none' : 'flex';
      
      if (window.innerWidth <= 992) {
        if (!isVisible) {
          mainNav.style.flexDirection = 'column';
          mainNav.style.position = 'absolute';
          mainNav.style.top = '100%';
          mainNav.style.left = '0';
          mainNav.style.width = '100%';
          mainNav.style.backgroundColor = 'white';
          mainNav.style.padding = '20px';
          mainNav.style.boxShadow = '0 10px 30px rgba(0,0,0,0.1)';
        }
      }
    });
    
    window.addEventListener('resize', () => {
      if (window.innerWidth > 992) {
        mainNav.style.display = 'flex';
        mainNav.style.flexDirection = 'row';
        mainNav.style.position = 'static';
        mainNav.style.backgroundColor = 'transparent';
        mainNav.style.padding = '0';
        mainNav.style.boxShadow = 'none';
      } else {
        mainNav.style.display = 'none';
      }
    });
  }
}

function initCounters() {
  const statElements = document.querySelectorAll('.stat-number');
  
  const observerOptions = {
    threshold: 0.5,
    rootMargin: '0px 0px -50px 0px'
  };
  
  const observer = new IntersectionObserver((entries) => {
    entries.forEach(entry => {
      if (entry.isIntersecting) {
        const element = entry.target;
        const targetValue = parseInt(element.textContent);
        const duration = 2000;
        
        animateCounter(element, 0, targetValue, duration);
        observer.unobserve(element);
      }
    });
  }, observerOptions);
  
  statElements.forEach(element => {
    observer.observe(element);
  });
}

function animateCounter(element, start, end, duration) {
  let startTime = null;
  
  const step = (timestamp) => {
    if (!startTime) startTime = timestamp;
    const progress = Math.min((timestamp - startTime) / duration, 1);
    const value = Math.floor(progress * (end - start) + start);
    
    element.textContent = value.toLocaleString();
    
    if (progress < 1) {
      window.requestAnimationFrame(step);
    }
  };
  
  window.requestAnimationFrame(step);
}

function initForms() {
  const donationForm = document.getElementById('donationForm');
  if (donationForm) {
    const amountSelect = document.getElementById('amount');
    const customAmountGroup = document.getElementById('customAmountGroup');
    
    if (amountSelect && customAmountGroup) {
      amountSelect.addEventListener('change', function() {
        customAmountGroup.style.display = this.value === 'custom' ? 'block' : 'none';
      });
    }
    
    donationForm.addEventListener('submit', function(e) {
      e.preventDefault();
      const formData = new FormData(this);
      const amount = formData.get('amount') === 'custom' 
        ? formData.get('customAmount') 
        : formData.get('amount');
      
      alert(`Thank you for your donation of $${amount}! This is a demo.`);
      donationForm.reset();
      if (customAmountGroup) customAmountGroup.style.display = 'none';
    });
  }
  
  const contactForm = document.getElementById('contactForm');
  if (contactForm) {
    contactForm.addEventListener('submit', function(e) {
      e.preventDefault();
      
      const name = document.getElementById('contactName').value.trim();
      const email = document.getElementById('contactEmail').value.trim();
      const message = document.getElementById('contactMessage').value.trim();
      
      if (!name || !email || !message) {
        alert('Please fill in all required fields.');
        return;
      }
      
      if (!isValidEmail(email)) {
        alert('Please enter a valid email address.');
        return;
      }
      
      alert('Thank you for your message! We will get back to you within 24-48 hours.');
      contactForm.reset();
    });
  }
}

function isValidEmail(email) {
  const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
  return emailRegex.test(email);
}

function initSmoothScroll() {
  document.querySelectorAll('a[href^="#"]').forEach(anchor => {
    anchor.addEventListener('click', function(e) {
      const href = this.getAttribute('href');
      
      if (href === '#' || href.includes('.html')) return;
      
      e.preventDefault();
      const targetElement = document.querySelector(href);
      
      if (targetElement) {
        window.scrollTo({
          top: targetElement.offsetTop - 100,
          behavior: 'smooth'
        });
      }
    });
  });
}

document.getElementById("contactForm").addEventListener("submit", async function (e) {
    e.preventDefault();

    const data = {
        name: document.getElementById("contactName").value,
        email: document.getElementById("contactEmail").value,
        phone: document.getElementById("contactPhone").value,
        subject: document.getElementById("contactSubject").value,
        message: document.getElementById("contactMessage").value,
        newsletter: document.getElementById("newsletter").checked
    };

    try {
        const res = await fetch("http://localhost:3000/api/contact", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            body: JSON.stringify(data)
        });

        const result = await res.json();
        alert(result.message);

        if (result.success) this.reset();
    } catch (err) {
        alert("Failed to send message");
    }
});
